import XCTest
@testable import death_app_Watch_App

/// Comprehensive boundary value and input validation testing
final class BoundaryValueTests: XCTestCase {
    
    var validator: InputValidator!
    var calculator: BoundaryCalculator!
    
    override func setUp() {
        super.setUp()
        validator = InputValidator()
        calculator = BoundaryCalculator()
    }
    
    override func tearDown() {
        validator = nil
        calculator = nil
        super.tearDown()
    }
    
    // MARK: - Age Boundary Tests
    
    func testAgeBoundaryValues() {
        // Test minimum valid age
        XCTAssertNoThrow(try validator.validateAge(0.0))
        
        // Test just below minimum
        XCTAssertThrowsError(try validator.validateAge(-0.1)) { error in
            XCTAssertTrue(error is ValidationError)
        }
        
        // Test maximum reasonable age
        XCTAssertNoThrow(try validator.validateAge(150.0))
        
        // Test just above maximum
        XCTAssertThrowsError(try validator.validateAge(150.1)) { error in
            XCTAssertTrue(error is ValidationError)
        }
        
        // Test extreme values
        XCTAssertThrowsError(try validator.validateAge(Double.infinity))
        XCTAssertThrowsError(try validator.validateAge(-Double.infinity))
        XCTAssertThrowsError(try validator.validateAge(Double.nan))
    }
    
    func testAgeCalculationPrecision() {
        // Test fractional ages
        let preciseAge = 25.123456789
        let result = calculator.calculateMortalityRisk(age: preciseAge)
        
        XCTAssertNotNil(result)
        XCTAssertTrue(result.age == preciseAge, "Should maintain age precision")
        
        // Test very small age differences
        let age1 = 30.0
        let age2 = 30.0000001
        
        let result1 = calculator.calculateMortalityRisk(age: age1)
        let result2 = calculator.calculateMortalityRisk(age: age2)
        
        // Results should be very close but not identical
        let difference = abs(result1.mortalityRate - result2.mortalityRate)
        XCTAssertLessThan(difference, 0.001, "Small age differences should produce small result differences")
    }
    
    // MARK: - Health Metric Boundary Tests
    
    func testHeartRateBoundaries() {
        // Test minimum viable heart rate
        XCTAssertNoThrow(try validator.validateHeartRate(30))
        
        // Test just below minimum
        XCTAssertThrowsError(try validator.validateHeartRate(29)) { error in
            XCTAssertTrue(error is HealthMetricValidationError)
        }
        
        // Test maximum viable heart rate
        XCTAssertNoThrow(try validator.validateHeartRate(220))
        
        // Test just above maximum
        XCTAssertThrowsError(try validator.validateHeartRate(221)) { error in
            XCTAssertTrue(error is HealthMetricValidationError)
        }
        
        // Test extreme values
        XCTAssertThrowsError(try validator.validateHeartRate(0))
        XCTAssertThrowsError(try validator.validateHeartRate(-10))
        XCTAssertThrowsError(try validator.validateHeartRate(1000))
    }
    
    func testBloodPressureBoundaries() {
        // Test minimum viable blood pressure
        XCTAssertNoThrow(try validator.validateBloodPressure(systolic: 70, diastolic: 40))
        
        // Test just below minimum
        XCTAssertThrowsError(try validator.validateBloodPressure(systolic: 69, diastolic: 40))
        XCTAssertThrowsError(try validator.validateBloodPressure(systolic: 70, diastolic: 39))
        
        // Test maximum viable blood pressure
        XCTAssertNoThrow(try validator.validateBloodPressure(systolic: 250, diastolic: 150))
        
        // Test just above maximum
        XCTAssertThrowsError(try validator.validateBloodPressure(systolic: 251, diastolic: 150))
        XCTAssertThrowsError(try validator.validateBloodPressure(systolic: 250, diastolic: 151))
        
        // Test invalid pressure relationships
        XCTAssertThrowsError(try validator.validateBloodPressure(systolic: 100, diastolic: 110)) { error in
            XCTAssertTrue(error is BloodPressureValidationError)
        }
    }
    
    func testExerciseTimeBoundaries() {
        // Test minimum exercise time
        XCTAssertNoThrow(try validator.validateExerciseMinutes(0))
        
        // Test negative exercise time
        XCTAssertThrowsError(try validator.validateExerciseMinutes(-1))
        
        // Test maximum reasonable exercise time (24 hours)
        XCTAssertNoThrow(try validator.validateExerciseMinutes(1440))
        
        // Test excessive exercise time
        XCTAssertThrowsError(try validator.validateExerciseMinutes(1441))
        
        // Test fractional minutes
        XCTAssertNoThrow(try validator.validateExerciseMinutes(30.5))
    }
    
    func testSleepHoursBoundaries() {
        // Test minimum sleep
        XCTAssertNoThrow(try validator.validateSleepHours(0))
        
        // Test negative sleep
        XCTAssertThrowsError(try validator.validateSleepHours(-0.1))
        
        // Test maximum sleep (24 hours)
        XCTAssertNoThrow(try validator.validateSleepHours(24))
        
        // Test excessive sleep
        XCTAssertThrowsError(try validator.validateSleepHours(24.1))
        
        // Test fractional hours
        XCTAssertNoThrow(try validator.validateSleepHours(7.5))
    }
    
    // MARK: - Weight and BMI Boundary Tests
    
    func testWeightBoundaries() {
        // Test minimum viable weight (in kg)
        XCTAssertNoThrow(try validator.validateWeight(1.0))
        
        // Test below minimum
        XCTAssertThrowsError(try validator.validateWeight(0.9))
        
        // Test maximum reasonable weight
        XCTAssertNoThrow(try validator.validateWeight(700.0))
        
        // Test excessive weight
        XCTAssertThrowsError(try validator.validateWeight(700.1))
        
        // Test fractional weights
        XCTAssertNoThrow(try validator.validateWeight(70.5))
    }
    
    func testHeightBoundaries() {
        // Test minimum viable height (in meters)
        XCTAssertNoThrow(try validator.validateHeight(0.5))
        
        // Test below minimum
        XCTAssertThrowsError(try validator.validateHeight(0.49))
        
        // Test maximum reasonable height
        XCTAssertNoThrow(try validator.validateHeight(3.0))
        
        // Test excessive height
        XCTAssertThrowsError(try validator.validateHeight(3.01))
        
        // Test fractional heights
        XCTAssertNoThrow(try validator.validateHeight(1.75))
    }
    
    func testBMICalculationBoundaries() {
        // Test extreme BMI scenarios
        let extremelyLowBMI = calculator.calculateBMI(weight: 30, height: 2.0) // BMI = 7.5
        XCTAssertLessThan(extremelyLowBMI, 15, "Should handle extremely low BMI")
        
        let extremelyHighBMI = calculator.calculateBMI(weight: 200, height: 1.0) // BMI = 200
        XCTAssertGreaterThan(extremelyHighBMI, 100, "Should handle extremely high BMI")
        
        // Test normal range boundaries
        let lowerNormalBMI = calculator.calculateBMI(weight: 54.4, height: 1.75) // BMI ≈ 17.8
        XCTAssertTrue(lowerNormalBMI > 17.5 && lowerNormalBMI < 18.5, "Should be at lower normal boundary")
        
        let upperNormalBMI = calculator.calculateBMI(weight: 76.6, height: 1.75) // BMI ≈ 25.0
        XCTAssertTrue(upperNormalBMI > 24.5 && upperNormalBMI < 25.5, "Should be at upper normal boundary")
    }
    
    // MARK: - Date and Time Boundary Tests
    
    func testDateBoundaries() {
        let validator = DateValidator()
        
        // Test birthdate boundaries
        let today = Date()
        let yesterday = Calendar.current.date(byAdding: .day, value: -1, to: today)!
        let farFuture = Calendar.current.date(byAdding: .year, value: 10, to: today)!
        let farPast = Calendar.current.date(byAdding: .year, value: -200, to: today)!
        
        // Valid birthdate (yesterday)
        XCTAssertNoThrow(try validator.validateBirthdate(yesterday))
        
        // Invalid birthdate (future)
        XCTAssertThrowsError(try validator.validateBirthdate(farFuture))
        
        // Invalid birthdate (too far in past)
        XCTAssertThrowsError(try validator.validateBirthdate(farPast))
    }
    
    func testTimeZoneBoundaries() {
        let timeZoneValidator = TimeZoneValidator()
        
        // Test extreme time zone offsets
        XCTAssertNoThrow(try timeZoneValidator.validateTimeZoneOffset(-12 * 3600)) // UTC-12
        XCTAssertNoThrow(try timeZoneValidator.validateTimeZoneOffset(14 * 3600))  // UTC+14
        
        // Test invalid time zone offsets
        XCTAssertThrowsError(try timeZoneValidator.validateTimeZoneOffset(-13 * 3600)) // UTC-13
        XCTAssertThrowsError(try timeZoneValidator.validateTimeZoneOffset(15 * 3600))  // UTC+15
    }
    
    // MARK: - String Length Boundary Tests
    
    func testStringLengthBoundaries() {
        let stringValidator = StringValidator()
        
        // Test empty strings
        XCTAssertThrowsError(try stringValidator.validateUserName(""))
        
        // Test single character
        XCTAssertNoThrow(try stringValidator.validateUserName("A"))
        
        // Test maximum length
        let maxLengthString = String(repeating: "A", count: 50)
        XCTAssertNoThrow(try stringValidator.validateUserName(maxLengthString))
        
        // Test over maximum length
        let overMaxString = String(repeating: "A", count: 51)
        XCTAssertThrowsError(try stringValidator.validateUserName(overMaxString))
        
        // Test special characters at boundaries
        XCTAssertNoThrow(try stringValidator.validateUserName("User123"))
        XCTAssertThrowsError(try stringValidator.validateUserName("User@123")) // Invalid character
    }
    
    // MARK: - Floating Point Precision Tests
    
    func testFloatingPointPrecision() {
        // Test very small differences
        let value1 = 1.0
        let value2 = 1.0 + Double.ulpOfOne
        
        let result1 = calculator.performPreciseCalculation(value1)
        let result2 = calculator.performPreciseCalculation(value2)
        
        XCTAssertNotEqual(result1, result2, "Should detect smallest possible difference")
        
        // Test precision loss scenarios
        let largeNumber = 1e15
        let smallAddition = 1.0
        let sum = largeNumber + smallAddition
        
        XCTAssertNotEqual(sum, largeNumber, "Should maintain precision in large number arithmetic")
    }
    
    func testFloatingPointEdgeCases() {
        // Test subnormal numbers
        let subnormal = Double.leastNormalMagnitude / 2
        XCTAssertNoThrow(try calculator.processSmallValue(subnormal))
        
        // Test very large numbers
        let veryLarge = Double.greatestFiniteMagnitude
        XCTAssertNoThrow(try calculator.processLargeValue(veryLarge))
        
        // Test special values
        XCTAssertThrowsError(try calculator.processSpecialValue(Double.infinity))
        XCTAssertThrowsError(try calculator.processSpecialValue(-Double.infinity))
        XCTAssertThrowsError(try calculator.processSpecialValue(Double.nan))
        
        // Test zero boundaries
        XCTAssertNoThrow(try calculator.processZeroValue(0.0))
        XCTAssertNoThrow(try calculator.processZeroValue(-0.0))
    }
    
    // MARK: - Array and Collection Boundary Tests
    
    func testCollectionBoundaries() {
        let collectionValidator = CollectionValidator()
        
        // Test empty collections
        XCTAssertThrowsError(try collectionValidator.validateHealthDataPoints([]))
        
        // Test single item
        let singleItem = [HealthDataPoint(timestamp: Date(), value: 70.0)]
        XCTAssertNoThrow(try collectionValidator.validateHealthDataPoints(singleItem))
        
        // Test maximum collection size
        let maxItems = Array(repeating: HealthDataPoint(timestamp: Date(), value: 70.0), count: 10000)
        XCTAssertNoThrow(try collectionValidator.validateHealthDataPoints(maxItems))
        
        // Test over maximum collection size
        let tooManyItems = Array(repeating: HealthDataPoint(timestamp: Date(), value: 70.0), count: 10001)
        XCTAssertThrowsError(try collectionValidator.validateHealthDataPoints(tooManyItems))
    }
    
    func testArrayIndexBoundaries() {
        let dataArray = [1, 2, 3, 4, 5]
        let arrayProcessor = ArrayProcessor()
        
        // Test valid indices
        XCTAssertNoThrow(try arrayProcessor.getElement(from: dataArray, at: 0))
        XCTAssertNoThrow(try arrayProcessor.getElement(from: dataArray, at: 4))
        
        // Test invalid indices
        XCTAssertThrowsError(try arrayProcessor.getElement(from: dataArray, at: -1))
        XCTAssertThrowsError(try arrayProcessor.getElement(from: dataArray, at: 5))
        
        // Test boundary conditions
        let element = try? arrayProcessor.getElement(from: dataArray, at: 4)
        XCTAssertEqual(element, 5, "Should get last element")
    }
    
    // MARK: - Currency and Financial Boundary Tests
    
    func testCurrencyBoundaries() {
        let currencyValidator = CurrencyValidator()
        
        // Test minimum monetary value
        XCTAssertNoThrow(try currencyValidator.validateAmount(0.01))
        
        // Test below minimum
        XCTAssertThrowsError(try currencyValidator.validateAmount(0.009))
        
        // Test maximum reasonable amount
        XCTAssertNoThrow(try currencyValidator.validateAmount(999999999.99))
        
        // Test over maximum
        XCTAssertThrowsError(try currencyValidator.validateAmount(1000000000.00))
        
        // Test precision
        XCTAssertNoThrow(try currencyValidator.validateAmount(123.45))
        XCTAssertThrowsError(try currencyValidator.validateAmount(123.456)) // Too many decimal places
    }
}

// MARK: - Mock Classes and Validators

class InputValidator {
    func validateAge(_ age: Double) throws {
        guard !age.isNaN && !age.isInfinite else {
            throw ValidationError.invalidNumber
        }
        guard age >= 0 && age <= 150 else {
            throw ValidationError.outOfRange
        }
    }
    
    func validateHeartRate(_ heartRate: Int) throws {
        guard heartRate >= 30 && heartRate <= 220 else {
            throw HealthMetricValidationError.heartRateOutOfRange
        }
    }
    
    func validateBloodPressure(systolic: Int, diastolic: Int) throws {
        guard systolic >= 70 && systolic <= 250 else {
            throw BloodPressureValidationError.systolicOutOfRange
        }
        guard diastolic >= 40 && diastolic <= 150 else {
            throw BloodPressureValidationError.diastolicOutOfRange
        }
        guard systolic > diastolic else {
            throw BloodPressureValidationError.invalidRelationship
        }
    }
    
    func validateExerciseMinutes(_ minutes: Double) throws {
        guard minutes >= 0 && minutes <= 1440 else {
            throw ValidationError.outOfRange
        }
    }
    
    func validateSleepHours(_ hours: Double) throws {
        guard hours >= 0 && hours <= 24 else {
            throw ValidationError.outOfRange
        }
    }
    
    func validateWeight(_ weight: Double) throws {
        guard weight >= 1.0 && weight <= 700.0 else {
            throw ValidationError.outOfRange
        }
    }
    
    func validateHeight(_ height: Double) throws {
        guard height >= 0.5 && height <= 3.0 else {
            throw ValidationError.outOfRange
        }
    }
}

class BoundaryCalculator {
    func calculateMortalityRisk(age: Double) -> MortalityResult {
        let mortalityRate = age * 0.01 // Simplified calculation
        return MortalityResult(age: age, mortalityRate: mortalityRate)
    }
    
    func calculateBMI(weight: Double, height: Double) -> Double {
        return weight / (height * height)
    }
    
    func performPreciseCalculation(_ value: Double) -> Double {
        return value * 1.000000001 // Precise multiplication
    }
    
    func processSmallValue(_ value: Double) throws {
        guard value.isFinite && value.isNormal || value == 0.0 else {
            throw ValidationError.invalidNumber
        }
    }
    
    func processLargeValue(_ value: Double) throws {
        guard value.isFinite else {
            throw ValidationError.invalidNumber
        }
    }
    
    func processSpecialValue(_ value: Double) throws {
        guard value.isFinite && !value.isNaN else {
            throw ValidationError.invalidNumber
        }
    }
    
    func processZeroValue(_ value: Double) throws {
        // Zero values are always valid
    }
    
    var lastCalculationTime: TimeInterval = 0
}

// MARK: - Additional Validator Classes

class DateValidator {
    func validateBirthdate(_ date: Date) throws {
        let now = Date()
        let minDate = Calendar.current.date(byAdding: .year, value: -150, to: now)!
        
        guard date <= now && date >= minDate else {
            throw ValidationError.dateOutOfRange
        }
    }
}

class TimeZoneValidator {
    func validateTimeZoneOffset(_ offset: TimeInterval) throws {
        let minOffset: TimeInterval = -12 * 3600 // UTC-12
        let maxOffset: TimeInterval = 14 * 3600  // UTC+14
        
        guard offset >= minOffset && offset <= maxOffset else {
            throw ValidationError.timeZoneOutOfRange
        }
    }
}

class StringValidator {
    func validateUserName(_ name: String) throws {
        guard !name.isEmpty && name.count <= 50 else {
            throw ValidationError.stringLengthInvalid
        }
        
        let allowedCharacters = CharacterSet.alphanumerics
        guard name.rangeOfCharacter(from: allowedCharacters.inverted) == nil else {
            throw ValidationError.invalidCharacters
        }
    }
}

class CollectionValidator {
    func validateHealthDataPoints(_ points: [HealthDataPoint]) throws {
        guard !points.isEmpty && points.count <= 10000 else {
            throw ValidationError.collectionSizeInvalid
        }
    }
}

class ArrayProcessor {
    func getElement<T>(from array: [T], at index: Int) throws -> T {
        guard index >= 0 && index < array.count else {
            throw ValidationError.indexOutOfRange
        }
        return array[index]
    }
}

class CurrencyValidator {
    func validateAmount(_ amount: Double) throws {
        guard amount >= 0.01 && amount < 1000000000.00 else {
            throw ValidationError.currencyOutOfRange
        }
        
        // Check decimal places (max 2)
        let multiplied = amount * 100
        guard multiplied == floor(multiplied) else {
            throw ValidationError.tooManyDecimalPlaces
        }
    }
}

// MARK: - Result Types

struct MortalityResult {
    let age: Double
    let mortalityRate: Double
}

struct HealthDataPoint {
    let timestamp: Date
    let value: Double
}

// MARK: - Error Types

enum ValidationError: Error {
    case invalidNumber
    case outOfRange
    case dateOutOfRange
    case timeZoneOutOfRange
    case stringLengthInvalid
    case invalidCharacters
    case collectionSizeInvalid
    case indexOutOfRange
    case currencyOutOfRange
    case tooManyDecimalPlaces
}

enum HealthMetricValidationError: Error {
    case heartRateOutOfRange
    case exerciseTimeInvalid
    case sleepHoursInvalid
}

enum BloodPressureValidationError: Error {
    case systolicOutOfRange
    case diastolicOutOfRange
    case invalidRelationship
}